



/*********************************************************************************************************//**
 * @file    BC3603.h
 * @version $Rev:: 5827         $
 * @date    $Date:: 2022-08-30 #$
 * @brief   The header file of the RF library.
 *************************************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ************************************************************************************************************/

/* Define to prevent recursive inclusion -------------------------------------------------------------------*/
#ifndef __BC3603_H
#define __BC3603_H

#ifdef __cplusplus
 extern "C" {
#endif

/* Includes ------------------------------------------------------------------------------------*/
#include "ht32.h"

/** @addtogroup HT32F67232_Peripheral_Driver HT32F67232 Peripheral Driver
  * @{
  */

/** @addtogroup RF
  * @{
  */


/* Exported constants --------------------------------------------------------------------------*/
/** @defgroup RF_Exported_Constants RF exported constants
  * @{
  */
/*--------------------- Strobe command ---------------------------------------------------------*/
#define 	WRITE_REGS_CMD					0x40  	/* Register write command */
#define 	READ_REGS_CMD					0xC0  	/* Register read command */
#define 	REGS_BANK_CMD      			0x20  	/* Set Register Bank command */
#define 	WRITE_SYNCWORD_CMD  			0x10  	/* Write Sync-Word command */
#define 	READ_SYNCWORD_CMD   			0x90  	/* Read Sync-Word command */
#define 	WRITE_FIFO_CMD      			0x11  	/* Write TX FIFO command */
#define 	READ_FIFO_CMD       			0x91  	/* Read RX FIFO command */
#define 	SOFT_RESET_CMD      			0x08  	/* Software reset command */
#define 	REST_TX_POS_CMD     			0x09  	/* TX FIFO address pointer reset command */
#define 	REST_RX_POS_CMD     			0x89  	/* RX FIFO address pointer reset command */
#define 	DEEP_SLEEP_CMD      			0x0A  	/* Enter Deep Sleep mode */
#define 	IDLE_MODE_CMD       			0x0B  	/* Enter Idle mode */
#define 	LIGHT_SLEEP_CMD     			0x0C  	/* Enter Light Sleep mode */
#define 	STANDBY_MODE_CMD    			0x0D  	/* Enter Standby mode */
#define 	TX_MODE_CMD         			0x0E  	/* Enter TX mode */
#define 	RX_MODE_CMD         			0x8E  	/* Enter RX mode */
#define 	REGSADDR_MASK       			0x3F

#define	SYNCWORD_MAX_SIZE				8			/* sync word max 8bytes */
#define	FIFO_MAX_SIZE					64       /* FIFO size max 64bytes */

/*--------------------- Common Register Address-------------------------------------------------*/
#define CONFIG_REGS 						0x00		/* Configuration Control Register */
#define CLOCK_CTL_REGS              0x01  	/* Reset/Clock Control Register */
#define IRQ_CTL_REGS                0x02  	/* Interrupt Control Register */
#define IRQ_ENABLE_REGS             0x03  	/* Interrupt enable register */
#define IRQ_STATUS_REGS             0x04  	/* interrupt status register */
#define CLOCK_CTL2_REGS					0x05
#define GPIO12_CTL_REGS 				0x06		/* GPIO 1/2 control register */
#define GPIO34_CTL_REGS             0x07		/* GPIO 3/4 control register */
#define GPIO_PULL_UP_REGS           0x08  	/* SPI/GPIO pull high control register */
#define TX_FIFO_SA_REGS             0x09  	/* TX FIFO start address register */
#define FIFO_CTL2_REGS              0x0A 		/* FIFO control 2 regster */
#define PREAMBLE_LENG_REGS          0x0B 		/* Packet preamble length register */
#define PACKET_CTL2_REGS				0x0C     /* Packet Control Register 2 */
#define PACKET_CTL3_REGS				0x0D     /* Packet Control Register 3 */
#define WHITEN_CTL_REGS					0x0E     /* Data whitening control Register */
#define TX_DATA_LENG_REGS				0x0F     /* Tx Payload length Register */
#define RX_DATA_LENG_REGS				0x10     /* Rx Payload length Register */
#define TRX_MODE_DELAY_REGS			0x11     /* Packet Control Register 7 */
#define HEADER_ADDR0_REGS				0x12     /* Payload Header address b5~b0 */
#define HEADER_ADDR1_REGS				0x13     /* Payload Header address b13~b6 */
#define MODULATOR_CTL1_REGS 			0x14     /* Modulator Control Register 1 */
#define MODULATOR_CTL2_REGS			0x15     /* Modulator Control Register 2 */
#define MODULATOR_CTL3_REGS			0x16     /* Modulator Control Register 3 */
#define DEMOULATOR_CTL1_REGS			0x17     /* Demodulator Control Register 1 */
#define DEMOULATOR_CTL2_REGS			0x18     /* Demodulator Control Register 2 */
#define DEMOULATOR_CTL3_REGS			0x19     /* Demodulator Control Register 3 */
#define DEMOULATOR_CTL4_REGS			0x1A     /* Demodulator Control Register 4 */
#define DEMOULATOR_CTL5_REGS			0x1B     /* Demodulator Control Register 5 */
#define DEMOULATOR_CTL8_REGS			0x1E     /* Demodulator Control Register 8 */
#define DEMOULATOR_CTL9_REGS			0x1F     /* Demodulator Control Register 9 */
/*--------------------- Bank 0 Register Address-------------------------------------------------*/
#define OPERATION_CTL_REGS 			0x20     /* Operation Mode Control Register */
#define FRACTIONAL_N_DN_REGS 			0x22     /* Fractional-N Synthesizer Control Register 1 */
#define FRACTIONAL_N_DKL_REGS			0x23     /* Fractional-N Synthesizer Control Register 2 */
#define FRACTIONAL_N_DKM_REGS			0x24     /* Fractional-N Synthesizer Control Register 3 */
#define FRACTIONAL_N_DKH_REGS			0x25     /* Fractional-N Synthesizer Control Register 4 */
#define OMODE_STATUS_REGS				0x26     /* Status Register */
#define RSSI_CTL_REGS					0x28     /* RSSI Control Register */
#define RSSI_NEGDB_REGS					0x29     /* RSSI Control Register */
#define RSSI_SYNC_OK_REGS				0x2A     /* RSSI Control Register */
#define ATR_CONTROL_REGS				0x2B     /* Auto TX/RX Control Register 1 */
#define ATR_CYCLE_L_REGS				0x2C     /* Auto TX/RX Control Register 2 */
#define ATR_CYCLE_H_REGS				0x2D     /* Auto TX/RX Control Register 3 */
#define ATR_ACTIVE_L_REGS				0x2E     /* Auto TX/RX Control Register 4 */
#define ATR_EACTIVE_L_REGS				0x2F     /* Auto TX/RX Control Register 5 */
#define ATR_EACTIVE_H_REGS				0x30     /* Auto TX/RX Control Register 6 */
#define ARK_CONTROL_REGS				0x31     /* Auto TX/RX Control Register 7 */
#define ARK_ACTIVE_REGS					0x32     /* Auto TX/RX Control Register 8 */
#define ATRCT_L_REGS						0x33     /* Auto TX/RX Control Register 9 */
#define ATRCT_H_REGS						0x34     /* Auto TX/RX Control Register 10 */
#define ATR_ACTIVE_H_REGS				0x35     /* Auto TX/RX Control Register 11 */
#define PACKET_CTL10_REGS				0x36     /* Packet Control Register 10 */
#define PREAMBLE_PATL_REGS				0x37     /* Preamble pattern low byte */
#define PREAMBLE_PATH_REGS				0x38     /* Preamble pattern hight byte */
#define CRC_SEED_L_REGS					0x39     /* CRC seed low byte */
#define CRC_SEED_H_REGS					0x3A     /* CRC seed high byte */
#define PACKET_CTL15_REGS				0x3B     /* Packet Control Register 15 */
#define XO_CAP_CTL_REGS					0x3C     /* XO Control Register 1 */
#define XO_SEL_CTL_REGS					0x3D     /* XO Control Register 2 */
#define LIRC_CTL_REGS					0x3E     /* XO Control Register 3 */
#define TX_POWER_CTL_REGS				0x3F     /* TX Control Register */

/*--------------------- Bank 3 Register Address-------------------------------------------------*/
#define	EFUSE_CTL0_REGS				0x30		/* eFuse Control Register 0 */
#define	EFUSE_CTL1_REGS				0x31     /* eFuse Control Register 1 */
#define	EFUSE_ADDR_REGS				0x36     /* eFuse address Register */
#define	EFUSE_DATA_REGS				0x37     /* eFuse data Register */

/* Definitions of CFG register																						*/
#define	CFG_BANK_MASK					(u8)(3 << 0)
#define	CFG_DIR_EN					   (u8)(1 << 4)
#define	CFG_RXCON_EN					(u8)(1 << 5)
#define	CFG_AGC_EN						(u8)(1 << 6)
#define	CFG_OOK_EN						(u8)(1 << 7)

/* Definitions of RC register                                                                 	*/
#define	RC_RST_LL						(u8)(1 << 0)
#define	RC_FSYCK_EN						(u8)(1 << 1)
#define	RC_FSYCK_DIV_MASK				(u8)(3 << 2)
#define	RC_XCLK_EN						(u8)(1 << 4)
#define	RC_XCLK_RDY						(u8)(1 << 5)
#define	RC_FSYCK_RDY					(u8)(1 << 6)
#define	RC_PWRON							(u8)(1 << 7)

/* Definitions of IRQ register                                                                 	*/
#define	IRQ_POLARITY					(u8)(1 << 0)
#define	IRQ_CLR_POR						(u8)(1 << 1)
#define	IRQ_RXDETS_MASK				(u8)(3 << 2)
#define	IRQ_RXCRCF						(u8)(1 << 4)
#define	IRQ_RXFFOW						(u8)(1 << 6)
#define	IRQ_RXTO							(u8)(1 << 7)
/* Definitions of Interrupt control register                                                   	*/
#define	IRQ_TXCMPIE						(u8)(1 << 0)
#define	IRQ_RXCMPIE						(u8)(1 << 1)
#define	IRQ_CALCMPIE					(u8)(1 << 2)
#define	IRQ_RXDETIE						(u8)(1 << 3)
#define	IRQ_RXERRIE						(u8)(1 << 4)
#define	IRQ_FIFOLTIE					(u8)(1 << 5)
#define	IRQ_ATRCTIE						(u8)(1 << 6)
#define	IRQ_ARKTFIE						(u8)(1 << 7)
/* Definitions of Interrupt status register                                                   	*/
#define	IRQ_TXCMPIF						(u8)(1 << 0)
#define	IRQ_RXCMPIF						(u8)(1 << 1)
#define	IRQ_CALCMPIF					(u8)(1 << 2)
#define	IRQ_RXDETIF						(u8)(1 << 3)
#define	IRQ_RXERRIF						(u8)(1 << 4)
#define	IRQ_FIFOLTIF					(u8)(1 << 5)
#define	IRQ_ATRCTIF						(u8)(1 << 6)
#define	IRQ_ARKTFIF						(u8)(1 << 7)
/* Definitions of IO control register                                                   			*/
#define	IO_GIO1S_MASK					(u8)(7 << 0)
#define	IO_GIO2S_MASK					(u8)(7 << 3)
#define	IO_PADDS_MASK					(u8)(3 << 6)
#define	IO_GIO3S_MASK					(u8)(0x0F << 0)
#define	IO_GIO4S_MASK					(u8)(0x0F << 4)
#define	IO_GIO1PU						(u8)(1 << 1)
#define	IO_GIO2PU						(u8)(1 << 2)
#define	IO_GIO3PU						(u8)(1 << 3)
#define	IO_GIO4PU						(u8)(1 << 4)
#define	IO_SPIPU							(u8)(1 << 6)
#define	IO_SDO_TEN						(u8)(1 << 7)
/* Definitions of FIFO control register                                                   		*/
#define	FIFO_FFMG_MASK					(u8)(3 << 0)
#define	FIFO_FFMG_EN					(u8)(1 << 2)
#define	FIFO_FFINF_EN					(u8)(1 << 3)
#define	FIFO_RXPL2F_EN					(u8)(1 << 4)
/* Definitions of Packet control register                                                   		*/
#define	PKT_SYNCLEN_MASK				(u8)(3 << 2)
#define	PKT_WHTFMT0						(u8)(1 << 4)
#define	PKT_TRAILER_EN					(u8)(1 << 5)
#define	PKT_TXPID_MASK					(u8)(3 << 6)

#define	PKT_PLHF_EN						(u8)(1 << 0)
#define	PKT_PLHLEN						(u8)(1 << 1)
#define	PKT_PLHAC_EN					(u8)(1 << 2)
#define	PKT_PLLEN_EN					(u8)(1 << 3)
#define	PKT_CRCFMT						(u8)(1 << 4)
#define	PKT_CRC_EN						(u8)(1 << 5)
#define	PKT_FEC_EN						(u8)(1 << 6)
#define	PKT_MCH_EN						(u8)(1 << 7)

#define	PKT_WHTSD_MASK					(u8)(0x7F << 0)
#define	PKT_WHT_EN						(u8)(1 << 7)

#define	PKT_DLYTXS_MASK				(u8)(7 << 0)
#define	PKT_DLYRXS_MASK				(u8)(7 << 3)
#define	PKT_RXPID_MASK					(u8)(3 << 6)

#define	PKT_PMLP_EN						(u8)(1 << 0)
#define	PKT_PMLPLEN						(u8)(1 << 1)
#define	PKT_SYNCLENLB					(u8)(1 << 2)
#define	PKT_CRCINV						(u8)(1 << 3)
#define	PKT_CRCBITO						(u8)(1 << 4)
#define	PKT_CRCBYTEO					(u8)(1 << 5)
#define	PKT_WHTFMT1						(u8)(1 << 6)

#define	PKT_OOKDT_EN					(u8)(1 << 0)
#define	PKT_OOKDT_POR					(u8)(1 << 1)
#define	PKT_OOKDT_MASK					(u8)(0x0F << 2)
#define	PKT_WHTSD87_MASK				(u8)(3 << 6)

/* Definitions of Modulator control register                                                   	*/
#define	MOD_DTR8							(u8)(1 << 0)
#define	MOD_DITHER_MASK				(u8)(3 << 2)
#define	MOD_RXIFOSHH_MASK				(u8)(0x0F << 4)

/* Definitions of Demodulator Control register                                               	*/
#define	DM2_CFO_EN0						(u8)(1 << 6)
#define	DM2_CFO_EN1						(u8)(1 << 7)
#define	DM4_THOLD_MASK					(u8)(0x0F << 4)

/* Definitions of Operation Mode control register                                              	*/
#define	OM_SX_EN							(u8)(1 << 0)
#define	OM_RTX_SEL						(u8)(1 << 1)
#define	OM_RTX_EN						(u8)(1 << 2)
#define	OM_ACAL_EN						(u8)(1 << 3)
#define	OM_BAND_MASK					(u8)(3 << 5)

/* Definitions of Status control register                                              			*/
#define	STA_OMST_MASK					(u8)(7 << 0)
#define	STA_CD_FLAG						(u8)(1 << 4)

/* Definitions of Auto TX/RX control register                                            			*/
#define	ATR_ENABLE						(u8)(1 << 0)
#define	ATR_MODE_MASK					(u8)(3 << 1)
#define	ATR_CTMS							(u8)(1 << 3)
#define	ATR_TUNIT						(u8)(1 << 4)
#define	ATR_CLKSS						(u8)(1 << 5)
#define	ATR_CLKDIV_MASK				(u8)(3 << 6)
#define	ATR_COMPATIBLE					(u8)(1 << 7)

/* Definitions of Auto packet repeat control register                                    			*/
#define	ARK_ENABLE						(u8)(1 << 0)
#define	ATR_WDLY_MASK					(u8)(3 << 1)
#define	ARK_RPNM_MASK					(u8)(0x0F << 4)

/* Definitions of OOK duty tune,packet control register15                       						*/
#define	OOKDT_ENABLE					(u8)(1 << 0)
#define	OOKDT_POLARITY					(u8)(1 << 1)
#define	OOKDT_TUNE_MASK				(u8)(0x0F << 2)

/* Definitions of XO control register                                    								*/
#define	XO_TRIM_MASK					(u8)(0x1F << 0)
#define	XO_SHIFT_MASK					(u8)(3 << 6)
#define	XO_DIV2							(u8)(1 << 3)
#define	XO_EFUSE_TRIM              (u8)(1 << 4)

/* Definitions of LIRC control register                                   								*/
#define	LIRC_ENABLE						(u8)(1 << 0)
#define	LIRC_TRIM_MASK					(u8)(0x1F << 1)
#define	LIRC_OVERWRITE					(u8)(1 << 6)
#define	LIRC_CAL_EN						(u8)(1 << 7)

/* Definitions of eFuse control register                                 								*/
#define	EFC_EN							(u8)(1 << 0)
#define	EFC_RW							(u8)(1 << 1)
#define	EFC_CHECK						(u8)(1 << 2)
#define	EFC_OW							(u8)(1 << 4)
#define	EFC_XOTS							(u8)(1 << 5)
#define	EFC_READY						(u8)(1 << 6)
#define	EFC_PDCMP						(u8)(1 << 7)

/**
  * @}
  */


/* Exported types ------------------------------------------------------------------------------------------*/
/** @defgroup RF_Exported_Types RF exported types
  * @{
  */
  
/**
  *  @brief Enumeration of Register Bank Control
  */
typedef enum
{
	REGS_BANK0 	=(u8)0,					/* register bank 0 */
	REGS_BANK1	=(u8)1,              /* register bank 1 */
	REGS_BANK2	=(u8)2,              /* register bank 2 */
	REGS_BANK3	=(u8)3               /* register bank 3 */	
} 	RF_BANK_Enum;  

/**
  *  @brief Enumeration of FSYCK clock(XCLK divison)
  */
typedef enum
{
	FSYCK_DIV1 	=(u8)0,					/* FSYCK = XCLK divison 1 */
	FSYCK_DIV2	=(u8)1,					/* FSYCK = XCLK divison 2 */
	FSYCK_DIV4	=(u8)2,					/* FSYCK = XCLK divison 4 */
	FSYCK_DIV8	=(u8)3					/* FSYCK = XCLK divison 8 */
} 	RF_FSYCK_Enum;  

/**
  *  @brief Enumeration of IRQ active & clear polarity
  */
typedef enum
{
	IRQ_INVERTED 		= DISABLE,		/* IRQ polarity is active low or falling edge */
	IRQ_NONINVERTED	= ENABLE			/* IRQ polarity is active high or rising edge */
} 	RF_IRQPOL_Enum;  

/**
  *  @brief Enumeration of RX detect event category
  */
typedef enum
{
	DET_CARRY 		=(u8)0,				/* RX detect event of carry */
	DET_SYNCWORD	=(u8)2				/* Rx detect event of sync-word */
} 	RF_RXDETS_Enum;  

/**
  *  @brief Enumeration of IO1 Register:GIO current Control
  */
typedef enum
{
	RF_GDCS500uA	=(u8)0,				/* GIO current strength 0.5mA */
	RF_GDCS1mA		=(u8)1,  			/* GIO current strength 1mA */
	RF_GDCS5mA		=(u8)2,  			/* GIO current strength 5mA */
	RF_GDCS10mA   	=(u8)3				/* GIO current strength 10mA */
} 	RF_GDCS_Enum;


/**
  *  @brief Enumeration of GIO1~4 pin function Control
  */
typedef enum
{
	RF_GIO1	=(u8)0,
	RF_GIO2	=(u8)1,
	RF_GIO3	=(u8)2,
	RF_GIO4	=(u8)3
}	RF_GIOS_Enum;

typedef enum
{
	RF_GIO_INPUT 	=(u8)0x00,    		/* GPIO input mode */
	RF_GIO_MISO		=(u8)0x01, 			/* 4-wire SPI data output */	
	RF_GIO_TRXD		=(u8)0x02,			/* Direct mode Tx/Rx data input/output */
	RF_GIO_IRQ 		=(u8)0x05,        /* interrupt request output */
	RF_GIO_LOSCI	=(u8)0x06,        /* ATR clock external input */
	RF_GIO_TRXDMO  =(u8)0x07,        /* Tx/Rx data monitor output */
	RF_GIO_TBCLK 	=(u8)0x08,        /* Tx clock output */
	RF_GIO_RBCLK	=(u8)0x09,       	/* Rx(recovery) clock output */
	RF_GIO_FCLKO	=(u8)0x0A,        /* Frequency clock output */
	RF_GIO_LIRCO	=(u8)0x0B,        /* internal LIRC clock output */
	RF_GIO_EPACO	=(u8)0x0C,			/* External PA control output */
	RF_GIO_ELNACO	=(u8)0x0D,			/* External LNA control output */
	RF_GIO_TRBCLK	=(u8)0x0E,			/* Direct mode Tx/Rx bit clock output */
	RF_GIO_OSCCTL	=(u8)0x0F         /* External OSC control output */
} 	RF_GIOF_Enum;

/**
  *  @brief Enumeration of FIFO length margin selection
  */
typedef enum
{
	FFMG4BYTE	=(u8)0,					/* FIFO length margin 4bytes */
	FFMG8BYTE	=(u8)1,              /* FIFO length margin 8bytes */
	FFMG16BYTE	=(u8)2,              /* FIFO length margin 16bytes */
	FFMG32BYTE  =(u8)3               /* FIFO length margin 32bytes */
} 	RF_FFMGS_Enum;


/**
  *  @brief Enumeration of preamble pattern and pattern width selection
  */
typedef enum
{
	PMSYNCMSB	=(u8)0,					/* preamble pattern form SYNCWORD MSB */
	PMPAT1BYTE	=(u8)1,              /* preamble pattern form PMLPAT register,pattern width 1 byte */
	PMPAT2BYTE	=(u8)3,              /* preamble pattern form PMLPAT register,pattern width 2 byte */
} 	RF_PMPATWD_Enum;

/**
  *  @brief Enumeration of TX/RX mode SYNCWORD length selection
  */
typedef enum
{
	SYNCLEN1BYTE	=(u8)1,				/* 1bytes SYNCWORD length */
	SYNCLEN2BYTE	=(u8)2,				/* 2bytes SYNCWORD length */
	SYNCLEN3BYTE	=(u8)3,				/* 3bytes SYNCWORD length */
	SYNCLEN4BYTE	=(u8)4,				/* 4bytes SYNCWORD length */
	SYNCLEN5BYTE	=(u8)5,				/* 5bytes SYNCWORD length */
	SYNCLEN6BYTE	=(u8)6,				/* 6bytes SYNCWORD length */
	SYNCLEN7BYTE	=(u8)7,				/* 7bytes SYNCWORD length */
	SYNCLEN8BYTE	=(u8)8				/* 8bytes SYNCWORD length */	
} 	RF_SYNCLEN_Enum;

/**
  *  @brief Enumeration of Whitening format selection
  */
typedef enum
{
	WHT_OFF 		=(u8)0,					/* whitening off */
	WHT36X		=(u8)1,					/* format x7+x6+x5+x4+1 */
	WHTPN7		=(u8)2,					/* format x7+x4+1 */
	WHTPN9CCIT	=(u8)3,					/* format x9+x5+1 */
	WHTPN9IBM	=(u8)4               /* format x9+x5+1 */
} 	RF_WHTFMT_Enum;

/**
  *  @brief Enumeration of CRC format selection
  */
typedef enum
{
	CRC_OFF 		=(u8)0,          		/* off CRC field  */
	CRC_CCIT16	=(u8)1,					/* CCIT16 format x16+x12+x5+1 */
	CRC_IBC16	=(u8)2					/* IBC16 format x16+x15+x2+1 */
} 	RF_CRCFMT_Enum;

typedef enum
{
	CRC_NONINVERTED	=(u8)0,			/* CRC no-inverted */	
	CRC_INVERTED 		=(u8)1			/* CRC inverted */
} 	RF_CRCPOL_Enum;  

typedef enum
{
	CRC_HIGH_FIRST		=(u8)0,			/* CRC high byte first */	
	CRC_LOW_FIRST 		=(u8)1			/* CRC low byte first */
} 	RF_BYTEOD_Enum;  

typedef enum
{
	CRC_MSB_FIRST		=(u8)0,			/* CRC high bit first */	
	CRC_LSB_FIRST 		=(u8)1			/* CRC low bit first */
} 	RF_BITOD_Enum;  


/**
  *  @brief Enumeration of Head field selection
  */
typedef enum
{
	HEAD_OFF 	=(u8)0,             	/* head field off */
	HEAD_8BIT	=(u8)1,					/* 8bit length head field */
	HEAD_16BIT	=(u8)2					/* 16bit length head field */
} 	RF_PLHF_Enum;

/**
  *  @brief Enumeration of RF Band selection
  */
typedef enum
{
	BAND315M	=(u8)0,						/* 315MHZ band */
	BAND433M	=(u8)1,						/* 433MHZ band */
	BAND470M	=(u8)2,						/* 470~510MHZ band */
	BAND868M	=(u8)3,						/* 868MHZ band */
	BAND915M	=(u8)4						/* 915MHZ band */	
} 	RF_BAND_Enum;

/**
  *  @brief Enumeration of Operation mode state indication
  */
typedef enum
{
	OMST_DEEPSLEEP		=(u8)0,			/* deep sleep mode */
	OMST_IDLE			=(u8)1,			/* idle mode */
	OMST_LIGHTSLEEP	=(u8)2,			/* light sleep mode */
	OMST_STANDBY		=(u8)3,			/* standby mode */
	OMST_TX				=(u8)4,			/* Tx mode */
	OMST_RX				=(u8)5,			/* Rx mode */
	OMST_CALIBRATION	=(u8)6,			/* calibration mode */
} 	RF_OMST_Enum;


/**
  *  @brief Enumeration of Auto TRX mode selection
  */
typedef enum
{
	ATRM_WOTS	=(u8)0,					/* auto wakeup on Tx mode-single */
	ATRM_WORS	=(u8)1,					/* auto wakeup on Rx mode-single */
	ATRM_WTMS	=(u8)2,					/* wakeup on timer mode-single */
	ATRM_WOTC 	=(u8)4,					/* auto wakeup on Tx mode-continuous */
	ATRM_WORC	=(u8)5,					/* auto wakeup on Rx mode-continuous */
	ATRM_WTMC	=(u8)6,					/* wakeup on timer mode-continuous */
} 	RF_ATRM_Enum;

/**
  *  @brief Enumeration of Auto TRX unit time selection
  */
typedef enum
{
	ATRTU_250US	= (u8)0,				/* unit:250us */
	ATRTU_1MS	= (u8)1					/* unit:1ms */
} 	RF_ATRTU_Enum;

/**
  *  @brief Enumeration of ATRCLK clock source selection
  */
typedef enum
{
	ATRCKS_LIRC 	= (u8)0,			/* clock source from internal LIRC clock */
	ATRCKS_EXTERAL	= (u8)1				/* Clock source from external clock (GIOn) */
} 	RF_ATRCKS_Enum;

/**
  *  @brief Enumeration of ATR clock frequency selection
  */
typedef enum
{
	ATRCK_DIV1 	=(u8)0,					/* clock source divide 1 */
	ATRCK_DIV4	=(u8)1,					/* clock source divide 4 */
	ATRCK_DIV8	=(u8)2,					/* clock source divide 8 */
	ATRCK_DIV16	=(u8)3					/* clock source divide 16 */	
} 	RF_ATRCKD_Enum;

/**
  *  @brief Enumeration of ATR clock frequency selection
  */
typedef enum
{
	ATRWDLY244	=(u8)0,					/* ATR wake up delay 244us */
	ATRWDLY488	=(u8)1,					/* ATR wake up delay 488us */
	ATRWDLY732	=(u8)2,					/* ATR wake up delay 732us */
	ATRWDLY976	=(u8)3					/* ATR wake up delay 976us */
} 	RF_ATRWDLY_Enum;

/**
 * @brief Adjust the Crystal Frequency
   */
#define XTAL_FREQUENCY    				(16000000UL)    				/* crystal frequency(unit:HZ) */

#define LIRC_FREQUENCY					(32768)							/* internal LIRC freqency(unit:HZ)*/

/**
 * @brief Adjust the Crystal (XCLK) Startup Timeout value
   */
#define XCLK_READY_TIME    			((uint32_t)0x000FFFFF)    	/*!< Time out for Crystal */

/**
 * @brief Adjust the VCO & RC calibration Timeout value
   */
#define VCAL_READY_TIME    			((uint32_t)0x000FFFFF)    	/*!< Time out for VCO & RC calibration */

/**
 * @brief Adjust the LIRC calibration Timeout value
   */
#define LCAL_READY_TIME    			((uint32_t)0x000FFFFF)    	/*!< Time out for LIRC calibration */

/**
 * @brief Minimum length to start infinite FIFO mode
   */
#define INFINITE_MIN_LENGTH			(192)    				/* unit:byte */


/**
 * @brief unit value for ATR time
   */
#define ATR_UNIT_250US					(250)
#define ATR_UNIT_1MS						(1000)

/**
 * @brief maximum value for ATR and ARK timer
   */
#define ATR_CYCLE_MAX					(65535)
#define ATR_RXACTIVE_MAX				(2047)
#define ATR_RXEXTEND_MAX				(65535)
#define ARK_REPEAT_MAX					(15)
#define ARK_RXACTIVE_MAX				(255)

/**
  *  @brief Enumeration of data rate selection
  */
typedef enum
{
	RECEIVER_SINGLE 	= DISABLE,					/* single mode */
	RECEIVER_CONTINUE = ENABLE      				/* continue mode */
}	RF_RCVM_Enum;
/**
  *  @brief Enumeration of data rate selection
  */
typedef enum
{
	DATARATE2K 		=(u8)0,							/* data rate 2KBps */
	DATARATE10K		=(u8)1,							/* data rate 10KBps */
	DATARATE50K		=(u8)2,							/* data rate 50KBps */
	DATARATE125K	=(u8)3,							/* data rate 125KBps */
	DATARATE250K	=(u8)4							/* data rate 250KBps */	
} 	RF_DATARATE_Enum;

/**
  *  @brief Enumeration of data rate selection
  */
typedef enum
{
	TXPOWER0DBM 	=(u8)0,							/* Tx Power 0dBm */
	TXPOWER10DBM 	=(u8)1,							/* Tx Power 10dBm */
	TXPOWER13DBM	=(u8)2,							/* Tx Power 13dBm */
	TXPOWER17DBM	=(u8)3,							/* Tx Power 17dBm */
	TXPOWER19DBM	=(u8)4							/* Tx Power 19dBm */	
} 	RF_TXPWR_Enum;


typedef struct
{
	u32					Frequency;              /* Frequency nuit:HZ */
	RF_BAND_Enum		RFBand;						/* RF band 315/433/470/868/915M */		
	RF_DATARATE_Enum	DataRate;					/* data rate 2/10/50/125/250Kbps */
	u8						TxPremLength;          	/* Tx preamble length */
	RF_PMPATWD_Enum	TxPremClass;				/* Tx preamble class */
	u16					TxPremPattern;          /* Tx preamble pattern */
	RF_SYNCLEN_Enum	SyncWordWidth;          /* syncword length 1~8bytes */
	ControlStatus		TrailerField;           /* payload trialer field enable/disable */
	RF_PLHF_Enum		HeadField;              /* payload head field enable/disable/length */
	ControlStatus		LengthField;            /* payload length field enable/disable */
	RF_CRCFMT_Enum		CRCField;               /* payload CRC field enable/disable/type */
	RF_CRCPOL_Enum		CRCInv;                 /* CRC data Inverse */
	RF_BYTEOD_Enum		CRCByteOD;              /* CRC byte Transmission order */
	RF_BITOD_Enum		CRCBitOD;               /* CRC bit Transmission order */
	u16					CRCSeed;                /* CRC seed */	
	ControlStatus		ManchesterCode;         /* manchester code enable/disable */
	ControlStatus		HammingCode;            /* hamming code enable/disable */
	ControlStatus		HeadAddrExamine;        /* examine head address */
	RF_WHTFMT_Enum		WhiteningCodec;        	/* whitening codec type */
	u16					WhiteningSeed;          /* whitening seed */
	RF_TXPWR_Enum		TxPower;	               /* Tx power */
	RF_RCVM_Enum		ReceiverMode;		      /* Rx mode single/continue */
	ControlStatus		AutomaticACK;           /* automatic acknowledge enable/disable */	
	RF_ATRTU_Enum		TimeUnit;					/* ATRRXAP & ATRRXEP & ARKRXAP time unit */
} RF_InitTypeDef;  

typedef struct
{
	RF_PMPATWD_Enum	PreambleClass;				/* preamble class select */
	u16					PreamblePattern;        /* preamble pattern */
	RF_SYNCLEN_Enum	SyncWord;              	/* syncword length 1~8bytes */
	ControlStatus		Trailer;                /* payload trialer field enable/disable */
	RF_PLHF_Enum		Head;                   /* payload head field enable/disable/length */
	ControlStatus		Length;                 /* payload length field enable/disable */
	RF_CRCFMT_Enum		CRC;                    /* payload CRC field enable/disable/type */
	RF_CRCPOL_Enum		CRCInv;                 /* CRC data Inverse */
	RF_BYTEOD_Enum		CRCByteOD;              /* CRC byte Transmission order */
	RF_BITOD_Enum		CRCBitOD;               /* CRC bit Transmission order */
	u16					CRCSeed;                /* CRC seed */
} RF_FieldTypeDef;

typedef struct
{
	ControlStatus		Manchester;            	/* manchester code enable/disable */
	ControlStatus		HammingCode;            /* hamming code enable/disable */
	ControlStatus		HeadAddrExamine;        /* examine head address */
	RF_WHTFMT_Enum		WhiteningCodec;        	/* whitening codec type */
	u16					WhiteningSeed;          /* whitening seed */	
} RF_CodecTypeDef;


typedef struct
{
	RF_ATRM_Enum		Mode;
	RF_ATRCKS_Enum		ClockSource;                  /* internal LIRC clock or external clock */
	RF_ATRCKD_Enum		Prescaler;                    /* Clock Prescaler Selection */ 
	u32					ClockFrequency;					/* external clock source frequency(unit:HZ) */
	u32					CycleCounter;                 /* cycle period time(unit:us) */
	RF_ATRWDLY_Enum	WakeUpDelayTime;
} ATR_InitTypeDef;

/**
  * @}
  */


/* Exported functions --------------------------------------------------------------------------------------*/
/** @defgroup RF_Exported_Functions RF exported functions
  * @{
  */  
ErrStatus BC3603_DeInit(void);
void 	BC3603_SetupSPI4Wires(void);
void 	BC3603_RegisterDefault(void);

void	BC3603_SoftwareReset(void);
void	BC3603_RegisterBank(RF_BANK_Enum Bank);
void	BC3603_EnterDeepSleep(void);
void	BC3603_EnterIdleMode(void);
void	BC3603_EnterLightSleep(void);
void	BC3603_EnterStandby(void);
void	BC3603_EnterTransmit(u8 FSAddr);
void	BC3603_EnterReceiver(void);
void	BC3603_ResetTxFIFOPosition(void);
void	BC3603_ResetRxFIFOPosition(void);
void	BC3603_SetSyncWord(u8 *SyncWord,u8 Length);
void	BC3603_GetSyncWord(u8 *SyncWord,u8 Length);
void 	BC3603_WriteTransmitPayload(u8 *Payload,u8 Length);
void	BC3603_WriteTxFIFO(u8 *FIFO,u8 Length);
u8 	BC3603_ReadReceiverPayload(u8 *Payload);
void	BC3603_ReadRxFIFO(u8 *FIFO,u8 Length);

ErrStatus BC3603_Configures(RF_InitTypeDef *RF_InitStruct);
void 	BC3603_DataRateConfig(RF_BAND_Enum Band,RF_DATARATE_Enum DataRate);
void 	BC3603_FrequencyConfig(RF_BAND_Enum Band,u32 Frequency);
void 	BC3603_SetTxPower(RF_BAND_Enum Band,RF_TXPWR_Enum Power);
void 	BC3603_SetPacketField(RF_FieldTypeDef *Field);
void 	BC3603_SetPacketCodec(RF_CodecTypeDef *Codec);
void 	BC3603_ReceiverConfig(RF_RCVM_Enum RxMode);
void 	BC3603_ExtendFIFOConfig(RF_FFMGS_Enum FFMgs,ControlStatus NewState);
void 	BC3603_InfiniteFIFOControl(ControlStatus NewState);
FlagStatus BC3603_GetInfiniteFIFOStatus(void);

void 	BC3603_GIOConfig(RF_GDCS_Enum GDCS,u8 PullBitMap,ControlStatus SDOTri);
void 	BC3603_SetGIOFunction(RF_GIOS_Enum GIO,RF_GIOF_Enum Fun);
void 	BC3603_SetCrystalCload(u8 xshift,u8 xtim);
ErrStatus BC3603_WaitCrystalReady(void);
FlagStatus BC3603_GetPowerOnFlag(void);
void 	BC3603_ClearPowerOnFlag(void);

void 	BC3603_SetFSYCKPrescaler(RF_FSYCK_Enum CKPRE,ControlStatus NewState);
void 	BC3603_SetReceiverDetectEvent(RF_RXDETS_Enum DETS);
FlagStatus BC3603_GetReceiverErrorStatus(u8 FlagBitMap);
u8 	BC3603_GetReceiverErrorFlag(void);
void 	BC3603_SetIRQPolarity(RF_IRQPOL_Enum POR,RF_IRQPOL_Enum CPOR);
void 	BC3603_IRQConfig(u8 IrqBitMap,ControlStatus NewState);
u8		BC3603_GetIRQFlag(void);
FlagStatus BC3603_GetIRQFlagStatus(u8 FlagBitMap);
void 	BC3603_ClearIRQFlag(u8 FlagBitMap);

void	BC3603_SetPreambleLength(u8 Length);
void	BC3603_SetTransmitLength(u8 Length);
void	BC3603_SetReceiverLength(u8 Length);
u8		BC3603_GetReceiverLength(void);
void 	BC3603_SetPacketID(u8 PID);
u8 	BC3603_GetPacketID(void);
void 	BC3603_SetHeaderAddress(u16 PLHA);
u16 	BC3603_GetHeaderAddress(void);
RF_OMST_Enum BC3603_GetOperationState(void);
FlagStatus BC3603_GetCarrierDetectStatus(void);
void 	BC3603_SetCarrierDetectThreshold(u8 THold);
u8 	BC3603_GetRSSIValue(void);
u8 	BC3603_GetSYNCRSSIValue(void);
void 	BC3603_ATRKTimeUnit(RF_ATRTU_Enum NewState);
void 	BC3603_ATRConfig(ATR_InitTypeDef *ATR_InitStruct);
void 	BC3603_SetWORPeriod(u32 RxActive,u32 RxExtend);
void 	BC3603_ATRControl(ControlStatus NewState);
u16 	BC3603_GetATRCycleValue(void);
u16 	BC3603_GetATRCycleCounter(void);
void 	BC3603_UpdateATRCycleCounter(u16 NewCounter);
void 	BC3603_ARKControl(ControlStatus NewState);
void 	BC3603_ARKRepeatPeriod(u8 Repeat,u32 Period);
u32 	BC3603_ATRConvertTimeUnit(u32 Period,u32 Max,float Unit);
void 	BC3603_OOKDutyTune(ControlStatus ODTE,FlagStatus ODTP, u8 ODTS);
ErrStatus BC3603_VCOAutoCalibration(void);
ErrStatus BC3603_LIRCCalibration(void);

/**
  * @}
  */


/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif

